import SwiftUI

struct SurveyDataExportView: View {
    @StateObject private var viewModel = SurveyViewModel()
    @Environment(\.dismiss) private var dismiss
    @State private var exportedData: String = ""
    @State private var showingExportSheet = false
    @State private var showingImportSheet = false
    @State private var importText: String = ""
    @State private var showingResetConfirmation = false
    @State private var showingAlert = false
    @State private var alertMessage = ""
    @State private var alertTitle = ""
    
    var body: some View {
        NavigationStack {
            ZStack {
                Color.black.ignoresSafeArea()
                
                ScrollView {
                    VStack(spacing: 20) {
                        // Header
                        VStack(spacing: 10) {
                            Image(systemName: "square.and.arrow.up.on.square")
                                .font(.largeTitle)
                                .foregroundColor(.blue)
                            
                            Text("Survey Data Management")
                                .font(.title2)
                                .fontWeight(.bold)
                                .foregroundColor(.white)
                            
                            Text("Export, import, or reset your survey data")
                                .font(.caption)
                                .foregroundColor(.secondary)
                                .multilineTextAlignment(.center)
                        }
                        .padding()
                        
                        // Current Data Status
                        if let surveyData = viewModel.surveyData.isCompleted ? viewModel.surveyData : nil {
                            currentDataStatusView(surveyData)
                        } else {
                            noDataView
                        }
                        
                        // Action Buttons
                        VStack(spacing: 15) {
                            Button("Export Survey Data") {
                                exportData()
                            }
                            .buttonStyle(.borderedProminent)
                            .tint(.blue)
                            .disabled(!viewModel.surveyData.isCompleted)
                            
                            Button("Import Survey Data") {
                                showingImportSheet = true
                            }
                            .buttonStyle(.bordered)
                            .tint(.green)
                            
                            Button("Reset Survey Data") {
                                showingResetConfirmation = true
                            }
                            .buttonStyle(.bordered)
                            .tint(.red)
                            .disabled(!viewModel.surveyData.isCompleted)
                        }
                        .padding()
                        
                        // Data Privacy Information
                        privacyInformationView
                    }
                }
            }
        }
        .navigationBarHidden(true)
        .onAppear {
            viewModel.startSurvey()
        }
        .sheet(isPresented: $showingExportSheet) {
            ExportDataSheet(exportedData: exportedData)
        }
        .sheet(isPresented: $showingImportSheet) {
            ImportDataSheet(
                importText: $importText,
                onImport: { importData() },
                onCancel: { showingImportSheet = false }
            )
        }
        .confirmationDialog("Reset Survey Data", isPresented: $showingResetConfirmation) {
            Button("Reset Data", role: .destructive) {
                resetData()
            }
            Button("Cancel", role: .cancel) { }
        } message: {
            Text("This will permanently delete all your survey data. This action cannot be undone.")
        }
        .alert(alertTitle, isPresented: $showingAlert) {
            Button("OK") { }
        } message: {
            Text(alertMessage)
        }
    }
    
    @ViewBuilder
    private func currentDataStatusView(_ surveyData: SurveyData) -> some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("Current Survey Data")
                .font(.headline)
                .foregroundColor(.white)
            
            VStack(alignment: .leading, spacing: 10) {
                DataStatusRow(label: "Last Updated", 
                             value: DateFormatter.shortDateTime.string(from: surveyData.lastUpdated))
                
                DataStatusRow(label: "Risk Level", 
                             value: surveyData.overallRiskLevel.displayName)
                
                DataStatusRow(label: "Smoking Status", 
                             value: surveyData.smokingStatus.displayName)
                
                DataStatusRow(label: "Exercise Frequency", 
                             value: surveyData.exerciseFrequency.displayName)
                
                DataStatusRow(label: "Data Complete", 
                             value: surveyData.isCompleted ? "Yes" : "No")
            }
        }
        .padding()
        .background(Color.gray.opacity(0.1))
        .cornerRadius(12)
        .padding(.horizontal)
    }
    
    private var noDataView: some View {
        VStack(spacing: 10) {
            Image(systemName: "doc.questionmark")
                .font(.largeTitle)
                .foregroundColor(.gray)
            
            Text("No Survey Data Found")
                .font(.headline)
                .foregroundColor(.white)
            
            Text("Complete the lifestyle survey first to access data management features")
                .font(.caption)
                .foregroundColor(.secondary)
                .multilineTextAlignment(.center)
        }
        .padding()
        .background(Color.gray.opacity(0.1))
        .cornerRadius(12)
        .padding(.horizontal)
    }
    
    private var privacyInformationView: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("Data Privacy")
                .font(.headline)
                .foregroundColor(.white)
            
            VStack(alignment: .leading, spacing: 8) {
                PrivacyInfoRow(icon: "lock.shield", 
                              title: "Encrypted Storage", 
                              description: "Your data is encrypted on device")
                
                PrivacyInfoRow(icon: "icloud.slash", 
                              title: "Local Only", 
                              description: "Data never leaves your device")
                
                PrivacyInfoRow(icon: "person.badge.shield.checkmark", 
                              title: "Full Control", 
                              description: "Export, import, or delete anytime")
            }
        }
        .padding()
        .background(Color.blue.opacity(0.1))
        .cornerRadius(12)
        .padding(.horizontal)
    }
    
    // MARK: - Actions
    
    private func exportData() {
        exportedData = viewModel.exportSurveyData()
        if !exportedData.isEmpty {
            showingExportSheet = true
        } else {
            alertTitle = "Export Failed"
            alertMessage = "Could not export survey data. Please try again."
            showingAlert = true
        }
    }
    
    private func importData() {
        let success = viewModel.importSurveyData(from: importText)
        showingImportSheet = false
        importText = ""
        
        if success {
            alertTitle = "Import Successful"
            alertMessage = "Survey data has been imported successfully."
        } else {
            alertTitle = "Import Failed"
            alertMessage = viewModel.validationErrors.first ?? "Invalid data format."
        }
        showingAlert = true
    }
    
    private func resetData() {
        viewModel.resetSurvey()
        alertTitle = "Data Reset"
        alertMessage = "All survey data has been cleared."
        showingAlert = true
    }
}

struct DataStatusRow: View {
    let label: String
    let value: String
    
    var body: some View {
        HStack {
            Text(label)
                .foregroundColor(.white)
                .font(.caption)
            
            Spacer()
            
            Text(value)
                .foregroundColor(.secondary)
                .font(.caption)
                .multilineTextAlignment(.trailing)
        }
    }
}

struct PrivacyInfoRow: View {
    let icon: String
    let title: String
    let description: String
    
    var body: some View {
        HStack(alignment: .top, spacing: 10) {
            Image(systemName: icon)
                .foregroundColor(.blue)
                .frame(width: 16)
            
            VStack(alignment: .leading, spacing: 2) {
                Text(title)
                    .font(.caption)
                    .fontWeight(.medium)
                    .foregroundColor(.white)
                
                Text(description)
                    .font(.caption2)
                    .foregroundColor(.secondary)
            }
            
            Spacer()
        }
    }
}

struct ExportDataSheet: View {
    let exportedData: String
    @Environment(\.dismiss) private var dismiss
    @State private var showingCopiedAlert = false
    
    var body: some View {
        NavigationStack {
            VStack(spacing: 20) {
                Text("Exported Survey Data")
                    .font(.headline)
                    .foregroundColor(.white)
                
                ScrollView {
                    Text(exportedData)
                        .font(.caption2)
                        .foregroundColor(.white)
                        .padding()
                        .background(Color.gray.opacity(0.2))
                        .cornerRadius(8)
                }
                
                HStack(spacing: 15) {
                    Button("Share") {
                        shareData()
                    }
                    .buttonStyle(.borderedProminent)
                    
                    Button("Done") {
                        dismiss()
                    }
                    .buttonStyle(.bordered)
                }
            }
            .padding()
            .background(Color.black)
        }
        .alert("Shared!", isPresented: $showingCopiedAlert) {
            Button("OK") { }
        } message: {
            Text("Survey data shared successfully")
        }
    }
    
    private func shareData() {
        // For watchOS, we'll use a simple alert to indicate the data is ready to be shared
        // In a real implementation, you might want to use Handoff or other watchOS sharing mechanisms
        DispatchQueue.main.async {
            self.showingCopiedAlert = true
        }
    }
}

struct ImportDataSheet: View {
    @Binding var importText: String
    let onImport: () -> Void
    let onCancel: () -> Void
    
    var body: some View {
        NavigationStack {
            VStack(spacing: 20) {
                Text("Import Survey Data")
                    .font(.headline)
                    .foregroundColor(.white)
                
                Text("Paste exported survey data below")
                    .font(.caption)
                    .foregroundColor(.secondary)
                
                ScrollView {
                    TextField("Paste survey data here...", text: $importText)
                        .font(.caption2)
                        .padding()
                        .background(Color.gray.opacity(0.2))
                        .cornerRadius(8)
                        .frame(minHeight: 120)
                }
                
                HStack(spacing: 15) {
                    Button("Cancel") {
                        onCancel()
                    }
                    .buttonStyle(.bordered)
                    
                    Button("Import") {
                        onImport()
                    }
                    .buttonStyle(.borderedProminent)
                    .disabled(importText.isEmpty)
                }
            }
            .padding()
            .background(Color.black)
        }
    }
}

// MARK: - Date Formatter Extension
extension DateFormatter {
    static let shortDateTime: DateFormatter = {
        let formatter = DateFormatter()
        formatter.dateStyle = .short
        formatter.timeStyle = .short
        return formatter
    }()
}

#Preview {
    SurveyDataExportView()
}